/*!
*******************************************************************************
* \file             spi_tclMySpinVrStream.cpp
* \brief            Copy audio data from an ALSA source to an ALSA sink
*******************************************************************************
\verbatim
PROJECT:        Gen3 Projects
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Copy audio data from an ALSA source to an ALSA sink
COPYRIGHT:      &copy; BSOT

HISTORY:
Date       |  Author                          | Modifications
08.10.2015 |  Tobias Quintern (BSOT/ENG)      | Initial version
05.01.2016 |  Tobias Quintern (BSOT/ENG)      | Changed public interface for increased flexibility

\endverbatim
******************************************************************************/





/******************************************************************************/
/* INCLUDES                                                                   */
/******************************************************************************/

#include "spi_tclMySpinVrStream.h"
#include "mspin_logging.h"


U32 spi_tclMySpinVrStream::m_u32InstanceCounter = 0;

S32 spi_tclMySpinVrStream::s32SetHwParams( snd_pcm_t *potrDeviceHandle ,
        const bool cobSetBufferSizeAndPeriodSize ,
        const snd_pcm_uframes_t cou32BufferSize ,
        const snd_pcm_uframes_t cou32PeriodSize )
{

    int err = 0;
    int dir = 0;

    unsigned int resample = 1;

    unsigned int rrate;
    snd_pcm_uframes_t rperiod_size = 0;
    snd_pcm_uframes_t rbuffer_size = 0;


    snd_pcm_hw_params_t *params = NULL;

    // Allocate invalid hw params using alloca
    // TBD: use snd_pcm_hw_params_malloc and return only at end of function
    snd_pcm_hw_params_alloca( &params );

    /* choose all parameters */
    err = snd_pcm_hw_params_any(potrDeviceHandle, params);
    if (err < 0) {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Broken configuration for playback: no configurations available: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
        return err;
    }

    /* set hardware resampling */
    err = snd_pcm_hw_params_set_rate_resample(potrDeviceHandle, params, resample);
    if (err < 0) {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Resampling setup failed for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
        return err;
    }

    /* set the interleaved read/write format */
    err = snd_pcm_hw_params_set_access(potrDeviceHandle, params, m_enAccess);
    if (err < 0) {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Access type not available for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
        return err;
    }
    /* set the sample format */
    err = snd_pcm_hw_params_set_format(potrDeviceHandle, params, m_enFormat);
    if (err < 0) {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Sample format not available for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
        return err;
    }
    /* set the count of channels */
    err = snd_pcm_hw_params_set_channels(potrDeviceHandle, params, m_u32Channels);
    if (err < 0) {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Channels count (%i) not available for playbacks: %s",__PRETTY_FUNCTION__, m_u32Channels, snd_strerror(err) );
        return err;
    }
    /* set the stream rate */
    rrate = m_u32Rate;
    err = snd_pcm_hw_params_set_rate_near(potrDeviceHandle, params, &rrate, 0);
    if (err < 0) {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Rate %iHz not available for playback: %s",__PRETTY_FUNCTION__, m_u32Rate, snd_strerror(err) );
        return err;
    }
    if (rrate != m_u32Rate) {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Rate doesn't match (requested %iHz, get %iHz)",__PRETTY_FUNCTION__, m_u32Rate, rrate );
        return -EINVAL;
    }

    if( false == cobSetBufferSizeAndPeriodSize )
    {
        U32 buffer_time = 0;
        U32 period_time = 0;
        err = snd_pcm_hw_params_get_buffer_time_max(params,
                &buffer_time, 0);

        if (err < 0) {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Unable to get buffer time max for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
            return err;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s() Got Buffer time: %i",__PRETTY_FUNCTION__, buffer_time );
            if( 500000 < buffer_time )
            {
                buffer_time = 500000;

            }

            period_time = buffer_time/4;

        }

        if( 0 < period_time )
        {
            err = snd_pcm_hw_params_set_period_time_near(potrDeviceHandle, params,
                    &period_time, 0);

            if (err < 0) {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Unable to set period time for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
                return err;
            }
        }

        if( 0 < buffer_time )
        {
            err = snd_pcm_hw_params_set_buffer_time_near(potrDeviceHandle, params,
                    &buffer_time, 0);
            if (err < 0) {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Unable to set buffer time for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
                return err;
            }
        }

        mspin_log_printLn(eMspinVerbosityInfo, "%s() Got Buffer time: %i, Period time: %i",__PRETTY_FUNCTION__, buffer_time, period_time );

    }
    else
    {

        rbuffer_size = cou32BufferSize;
        err = snd_pcm_hw_params_set_buffer_size_near(potrDeviceHandle, params, &rbuffer_size);
        if (err < 0) {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Unable to set buffer size for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
            return err;
        }
        if (rbuffer_size != cou32BufferSize) {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Buffer size doesn't match (requested %i, get %i)",__PRETTY_FUNCTION__, (int)cou32BufferSize, (int)rbuffer_size );
            return -EINVAL;
        }

        rperiod_size = cou32PeriodSize;
        err = snd_pcm_hw_params_set_period_size_near(potrDeviceHandle, params, &rperiod_size, &dir);
        if (err < 0) {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Unable to set period size for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
            return err;
        }

        if (rperiod_size != cou32PeriodSize) {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Period size doesn't match (requested %i, get %i)",__PRETTY_FUNCTION__, (int)cou32PeriodSize, (int)rperiod_size );
            return -EINVAL;
        }
    }

    /* write the parameters to device */
    err = snd_pcm_hw_params(potrDeviceHandle, params);
    if (err < 0) {
        mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Unable to set hw params for playback: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
        return err;
    }

    return 0;
}


/*
 *   Underrun and suspend recovery
 */
S32 spi_tclMySpinVrStream::s32XrunRecovery(snd_pcm_t *potrHandle, S32 err)
{

    S32 s32RetVal = 0;
    S32 s32R = 0;
    U32 u32ResumeTries = 0;
    U32 u32ResumeTriesMax = 3;

    if (-EPIPE == err )
    {    /* under-run */
        s32R = snd_pcm_prepare(potrHandle);
        if( 0 > s32R )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Can't recovery from underrun, prepare failed: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
            s32RetVal = -1;
        }
        else
        {
            s32RetVal = 0;
        }

    }
    else if( -ESTRPIPE == err )
    {
        s32R = -EAGAIN;
        while( ( 0 == s32RetVal ) &&
               ( -EAGAIN == s32R ) )
        {
            s32R = snd_pcm_resume(potrHandle);

            if( -EAGAIN == s32R )
            {
                sleep(1);
                u32ResumeTries++;

                if( u32ResumeTriesMax <= u32ResumeTries )
                {
                    mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to resume pcm after %i tries",__PRETTY_FUNCTION__, u32ResumeTries );
                    s32RetVal = -2;
                }
                else
                {
                    u32ResumeTries++;
                }
            }
        }


        if( 0 == s32RetVal )
        {
            s32R = snd_pcm_prepare(potrHandle);
            if( s32R < 0 )
            {
                mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Can't recovery from suspend, prepare failed: %s",__PRETTY_FUNCTION__, snd_strerror(err) );
                s32RetVal = -3;
            }
        }
    }
    else
    {
        s32R = snd_pcm_recover(potrHandle, err , 0);
    }
    return s32RetVal;
}

S32 spi_tclMySpinVrStream::s32WriteBuffer( U32 u32Id, snd_pcm_t *const potrDeviceHandle,  const U8 *const copu8ReadBuffer, const U32 cou32FramesToWrite )
{
    S32 s32RetVal = 0;
    S32 s32R = 0;

    const U8 *pu8TmpPtr = (const U8 *)copu8ReadBuffer;
    S32 s32FramesToWrite = cou32FramesToWrite;

    while( 0 < s32FramesToWrite )
    {

        // Write data to playback device
        s32R = snd_pcm_mmap_writei(potrDeviceHandle, pu8TmpPtr, s32FramesToWrite );

        // Data written, update state
        if( 0 <= s32R )
        {
            s32FramesToWrite -= s32R;
            pu8TmpPtr += s32R;
        }
        // EAGAIN - can be ignored
        else if( -EAGAIN == s32R )
        {
            // ignore eagain
            //ETG_TRACE_USR4(("spi_tclMySpinVrStream::s32WriteBuffer(%i): snd_pcm_writei() returned EAGAIN", u32Id ));
        }
        // Other errors have to be handled
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: snd_pcm_mmap_writei() returned error: %i",__PRETTY_FUNCTION__, u32Id, s32R  );
            mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: cou32FramesToWrite %i, u32FramesToWrite: %i",__PRETTY_FUNCTION__,u32Id,cou32FramesToWrite,cou32FramesToWrite );

            s32XrunRecovery(potrDeviceHandle, s32R);
            //s32R = snd_pcm_recover(potrDeviceHandle, s32R , 0);
            s32FramesToWrite = 0;  // Skip frame
        }

        // Wait, if not all data was written
        if( 0 != s32FramesToWrite )
        {
            // use poll to wait for next event on playback device
            s32R = snd_pcm_wait(potrDeviceHandle, 1000);

            if( 0 == s32R )
            {
                mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) Timeout",__PRETTY_FUNCTION__, u32Id );
                s32FramesToWrite = 0;
            }
            else if( 1 != s32R )
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: snd_pcm_wait() returned error: %i",__PRETTY_FUNCTION__, u32Id, s32R );

                s32XrunRecovery(potrDeviceHandle, s32R);
                //s32R = snd_pcm_recover(potrDeviceHandle, s32R , 0);
                s32FramesToWrite = 0; // Skip frame
            }
            else
            {

            }
        }
    } /* while( 0 < u32FramesToWrite ) */


    return s32RetVal;
}

S32 spi_tclMySpinVrStream::s32OpenAlsaDevice( vrDeviceConfig *porDeviceConfig , snd_pcm_t **popotrDeviceHandle ,  snd_pcm_stream_t enStreamType )
{
    S32 s32RetVal = 0;
    S32 s32R = 0;

    snd_pcm_t *potrDeviceHandle = NULL;

    snd_pcm_uframes_t *pu32BufferSize = NULL;
    snd_pcm_uframes_t *pu32PeriodSize = NULL;

    std::string szDeviceName;

    if( ( NULL == porDeviceConfig ) ||
        ( NULL == popotrDeviceHandle ) )
    {
        s32RetVal = -1;
    }

    if( 0 == s32RetVal )
    {
        pu32BufferSize = &porDeviceConfig->bufferSize;
        pu32PeriodSize = &porDeviceConfig->periodSize;
        szDeviceName   = porDeviceConfig->deviceName;
    }

    mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) ENTER. enStreamType: 0x%X szDeviceName: %s",__PRETTY_FUNCTION__, m_u32Id,enStreamType,szDeviceName.c_str() );

    if( 0 == s32RetVal )
    {
        s32R = snd_pcm_open(&potrDeviceHandle, szDeviceName.c_str() , enStreamType, 0);

        if( 0 > s32R ) {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Capture open error: %s",__PRETTY_FUNCTION__, snd_strerror(s32R) );
            potrDeviceHandle = NULL;
            s32RetVal = -2;
        }
    }


    if( ( 0 == s32RetVal ) &&
        ( false == porDeviceConfig->setBufferSizeAndPeriodSize ) )
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) call snd_pcm_set_params()",__PRETTY_FUNCTION__, m_u32Id );
        s32R = snd_pcm_set_params(potrDeviceHandle,
                m_enFormat,
                m_enAccess,
                m_u32Channels,
                m_u32Rate,
                1,
                500000);

        if( 0 > s32R )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Capture open error: %s",__PRETTY_FUNCTION__, snd_strerror(s32R) );
            s32RetVal = -3;
        }
    }



    if( ( 0 == s32RetVal ) &&
        ( true == porDeviceConfig->setBufferSizeAndPeriodSize ) )
    {

        mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) call s32SetHwParams()",__PRETTY_FUNCTION__, m_u32Id );
        s32R = s32SetHwParams( potrDeviceHandle , porDeviceConfig->setBufferSizeAndPeriodSize, *pu32BufferSize , *pu32PeriodSize );

        if( 0 != s32R )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: s32SetHwParams() error",__PRETTY_FUNCTION__ );
            s32RetVal = -3;
        }
    }



    if( ( 0 == s32RetVal ) &&
            ( false == porDeviceConfig->blocking ) )
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) call snd_pcm_nonblock()",__PRETTY_FUNCTION__, m_u32Id );
        s32R = snd_pcm_nonblock( potrDeviceHandle , 1 );
        if( 0 > s32R )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Capture nonblock err: %s",__PRETTY_FUNCTION__, snd_strerror(s32R) );
            s32RetVal = -4;
        }
    }

    /* Obtain buffer and period size of ALSA device */
    if( 0 == s32RetVal )
    {

        s32R = snd_pcm_get_params( potrDeviceHandle ,
                pu32BufferSize,
                pu32PeriodSize);

        if( 0 != s32R )
        {
            s32RetVal = -4;
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityInfo, "%s(%i)BufferSize: %i PeriodSize: %i",__PRETTY_FUNCTION__, m_u32Id, *pu32BufferSize , *pu32PeriodSize );
        }
    }


    if( 0 != s32RetVal )
    {
        vCloseAlsaDevice( &potrDeviceHandle );
    }
    else
    {
        *popotrDeviceHandle = potrDeviceHandle;
    }


    mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) EXIT: %i",__PRETTY_FUNCTION__, m_u32Id, s32RetVal );

    return s32RetVal;
}

void spi_tclMySpinVrStream::vCloseAlsaDevice( snd_pcm_t **m_popotrDeviceHandle )
{
    if( ( NULL != m_popotrDeviceHandle ) &&
            ( NULL != *m_popotrDeviceHandle ) )
    {
        // Drop remaining frames
        snd_pcm_drop(*m_popotrDeviceHandle);

        // Close device
        snd_pcm_close(*m_popotrDeviceHandle);
        *m_popotrDeviceHandle = NULL;
    }
}

void* spi_tclMySpinVrStream::povCopyThread(void *pArg)
{
    spi_tclMySpinVrStream *poMySpinAlsaStream = (spi_tclMySpinVrStream*)pArg;

    S32 s32RetVal = 0;
    S32 s32R = 0;

    U32 u32FrameNum = 4000;
    U32 u32BytesPerFrame = 0;
    U32 u32BufferSize = 0;
    U8 *au8Buffer = NULL;

    S32 s32FramesRead = 0;
    U32 u32Id = 0;

    if( NULL == poMySpinAlsaStream )
    {
        s32RetVal = -1;
    }
    else
    {
        u32Id = poMySpinAlsaStream->m_u32Id;
        u32FrameNum = poMySpinAlsaStream->m_rPlaybackDeviceConfig.periodSize;
    }

    mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) Start thread",__PRETTY_FUNCTION__, u32Id );

    // Allocate transfer buffer
    if( 0 == s32RetVal )
    {
        u32BytesPerFrame = (snd_pcm_format_width(poMySpinAlsaStream->m_enFormat) / 8) * poMySpinAlsaStream->m_u32Channels;
        u32BufferSize = u32FrameNum*u32BytesPerFrame;
        au8Buffer = (U8*)calloc( u32BufferSize , sizeof(U8)  );
        if( NULL == au8Buffer )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: Failed to allocate %i bytes!",__PRETTY_FUNCTION__, u32Id, u32BufferSize );
            s32RetVal = -2;
        }
    }

    // Start streaming
    while( ( 0 == s32RetVal ) &&
            ( true == poMySpinAlsaStream->m_bThreadRunning ) )
    {

        // Read data from capture stream
        s32R = snd_pcm_mmap_readi(poMySpinAlsaStream->m_potrCaptureHandle, au8Buffer, u32FrameNum );

        if( 0 < s32R )
        {
            s32FramesRead = s32R;
            s32R = poMySpinAlsaStream->s32WriteBuffer(u32Id, poMySpinAlsaStream->m_potrPlaybackHandle, au8Buffer , s32FramesRead );
            if( 0 != s32R )
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: Failed write %i bytes: %i",__PRETTY_FUNCTION__, u32Id, s32FramesRead, s32R );
            }
        } /* if( 0 < s32FramesRead ) */
        else if( -EAGAIN == s32R )
        {

            // use poll to wait for next event on capture stream
            s32R = snd_pcm_wait(poMySpinAlsaStream->m_potrCaptureHandle, 1000);

            if( 0 == s32R )
            {
                mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) Capture Timeout",__PRETTY_FUNCTION__, u32Id );
            }
            else if( 1 != s32R )
            {
                mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: snd_pcm_wait() returned error: %i",__PRETTY_FUNCTION__, u32Id, s32R );
                s32RetVal = -3;
            }
        }
        else
        {
            mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: snd_pcm_mmap_readi bytes() returned: %i",__PRETTY_FUNCTION__, u32Id, s32R );
        }
    } /* while( ( 0 == s32RetVal ) && ( true == poMySpinAlsaStream->m_bThreadRunning ) ) */

    mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) Exit loop: %i",__PRETTY_FUNCTION__, u32Id );

    if( NULL != poMySpinAlsaStream )
    {
        poMySpinAlsaStream->m_bThreadRunning = false;
    }

    if( NULL != au8Buffer )
    {
        free( au8Buffer );
        au8Buffer = NULL;
    }

    return pArg;

}

S32 spi_tclMySpinVrStream::s32StartVRStream( const vrDeviceConfig corCaptureDeviceConfig ,
                                   const vrDeviceConfig corPlaybackDeviceConfig ,
                                   const snd_pcm_access_t coenAccess ,
                                const snd_pcm_format_t coenFormat ,
                                const unsigned int cou32Channels ,
                                const unsigned int cou32Samplerate )
{
    S32 s32RetVal = 0;
    S32 s32R = 0;

    if( true == m_bThreadRunning )
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: Thread already running!",__PRETTY_FUNCTION__, m_u32Id );
        s32RetVal = -1;
    }

    if( 16 <= m_szThreadName.length() )
    {
        mspin_log_printLn(eMspinVerbosityError, "%s(%i) ERROR: Thread name too long!",__PRETTY_FUNCTION__, m_u32Id );
        s32RetVal = -2;
    }

    // Store given parameters in object
    if( 0 == s32RetVal )
    {
        m_rCaptureDeviceConfig = corCaptureDeviceConfig;
        m_rPlaybackDeviceConfig = corPlaybackDeviceConfig;
        m_enAccess = coenAccess;
        m_enFormat = coenFormat;
        m_u32Channels = cou32Channels;
        m_u32Rate = cou32Samplerate;
    }

    /* Open capture device */
    if( 0 == s32RetVal )
    {
        s32R = s32OpenAlsaDevice( &m_rCaptureDeviceConfig , &m_potrCaptureHandle ,  SND_PCM_STREAM_CAPTURE );
        if( 0 != s32R )
        {
            s32RetVal = -3;
        }
    }

    /* Open playback device */
    if( 0 == s32RetVal )
    {

        s32R = s32OpenAlsaDevice( &m_rPlaybackDeviceConfig , &m_potrPlaybackHandle ,  SND_PCM_STREAM_PLAYBACK );
        if( 0 != s32R )
        {
            s32RetVal = -4;
        }
    }

    /* Print buffer and period size of capture and playback device */
    if( 0 == s32RetVal )
    {
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) Capture BufferSize: %i PeriodSize: %i",__PRETTY_FUNCTION__, m_u32Id, m_rCaptureDeviceConfig.bufferSize , m_rCaptureDeviceConfig.periodSize );
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) Playback BufferSize: %i PeriodSize: %i",__PRETTY_FUNCTION__, m_u32Id, m_rPlaybackDeviceConfig.bufferSize , m_rPlaybackDeviceConfig.periodSize );
    }

    /* Create stream copy thread */
    if( 0 == s32RetVal )
    {
        m_bThreadRunning = true;
        s32R = pthread_create(&m_threadId, NULL, povCopyThread, this);
        if( 0 != s32R )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to start audio stream",__PRETTY_FUNCTION__ );
            s32RetVal = -5;
        }
    }

    /* Assign name to thread */
    if( 0 == s32RetVal )
    {
        s32R = pthread_setname_np(m_threadId, m_szThreadName.c_str() );

        if( 0 != s32R )
        {
            mspin_log_printLn(eMspinVerbosityError, "%s() ERROR: Failed to set thread name",__PRETTY_FUNCTION__ );
            s32RetVal = -6;
        }
    }



    return s32RetVal;
}

S32 spi_tclMySpinVrStream::s32StopVrStream()
{
    S32 s32RetVal = 0;

    mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) ENTER",__PRETTY_FUNCTION__, m_u32Id );
    m_bThreadRunning = false;

    if( 0 != m_threadId )
    {
        pthread_join(m_threadId, NULL);
        m_threadId = 0;
        mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) Thread joined",__PRETTY_FUNCTION__, m_u32Id );
    }


    if( NULL != m_potrCaptureHandle )
    {
        vCloseAlsaDevice( &m_potrCaptureHandle );
    }

    if( NULL != m_potrPlaybackHandle )
    {
        vCloseAlsaDevice( &m_potrPlaybackHandle );
    }
    mspin_log_printLn(eMspinVerbosityInfo, "%s(%i) EXIT",__PRETTY_FUNCTION__, m_u32Id );

    return s32RetVal;
}

void spi_tclMySpinVrStream::vRegisterCallbacks( U32 cou32Context , vrStreamCallbacks rMySpinAlsaStreamCallback )
{
    (void)cou32Context;            // currently unused

    m_rMySpinAlsaStreamCallbacks = rMySpinAlsaStreamCallback;
}

spi_tclMySpinVrStream::spi_tclMySpinVrStream()
: m_u32Id(m_u32InstanceCounter)
, m_szThreadName("MySpinVrSt")
, m_potrCaptureHandle(NULL)
, m_potrPlaybackHandle(NULL)
, m_bThreadRunning(false)
, m_threadId(0)
, m_enAccess(SND_PCM_ACCESS_MMAP_INTERLEAVED)
, m_enFormat(SND_PCM_FORMAT_S16_LE)
, m_u32Channels(1)
, m_u32Rate(16000)
{
    m_u32InstanceCounter++;

}

spi_tclMySpinVrStream::~spi_tclMySpinVrStream()
{
    s32StopVrStream();
}
